import { Request, Response } from 'express';
import { supabase, supabaseAdmin, USER_ROLES, ONBOARDING_STATUS, TABLES } from '@/config/supabase';
import { SignupRequest, LoginRequest, AuthResponse, ApiResponse } from '@/types';
import { vendorService } from '@/modules/vendors/service';

/**
 * Auth Controller - Handles authentication operations
 */
export class AuthController {
  /**
   * Register a new user
   * POST /api/auth/signup
   */
  async signup(req: Request, res: Response) {
    try {
      const { email, password, business_name }: SignupRequest = req.body;

      // Validate input
      if (!email || !password) {
        return res.status(400).json({
          success: false,
          error: 'Email and password are required'
        });
      }

      // Check if user already exists
      const { data: existingUser } = await supabase.auth.admin.listUsers();
      const userExists = existingUser.users?.some(user => user.email === email);
      if (userExists) {
        return res.status(409).json({
          success: false,
          error: 'User already exists'
        });
      }

      // Create user with Supabase Auth
      const { data: authData, error: authError } = await supabase.auth.admin.createUser({
        email,
        password,
        email_confirm: true, // Auto-confirm email for development
        user_metadata: {
          role: USER_ROLES.VENDOR
        }
      });

      if (authError || !authData.user) {
        console.error('Auth creation error:', authError);
        return res.status(500).json({
          success: false,
          error: 'Failed to create user account'
        });
      }

      // Create vendor record automatically
      const vendorData = {
        user_id: authData.user.id,
        email: authData.user.email!,
        business_name: business_name || null,
        onboarding_status: ONBOARDING_STATUS.INCOMPLETE,
        subscription_plan: 'free' // Default to free plan
      };

      const { data: vendor, error: vendorError } = await vendorService.createVendor(vendorData);

      if (vendorError || !vendor) {
        console.error('Vendor creation error:', vendorError);
        // Clean up auth user if vendor creation fails
        await supabase.auth.admin.deleteUser(authData.user.id);
        return res.status(500).json({
          success: false,
          error: 'Failed to create vendor profile'
        });
      }

      // Sign in to get session tokens
      const { data: sessionData, error: sessionError } = await supabase.auth.signInWithPassword({
        email,
        password
      });

      if (sessionError) {
        console.error('Session generation error:', sessionError);
        return res.status(500).json({
          success: false,
          error: 'Failed to generate session'
        });
      }

      const response: AuthResponse = {
        user: authData.user,
        vendor,
        access_token: sessionData.session?.access_token || '',
        refresh_token: sessionData.session?.refresh_token || ''
      };

      res.status(201).json({
        success: true,
        data: response,
        message: 'User registered successfully'
      });
      return;

    } catch (error) {
      console.error('Signup error:', error);
      res.status(500).json({
        success: false,
        error: 'Registration failed'
      });
      return;
    }
  }

  /**
   * Login user
   * POST /api/auth/login
   */
  async login(req: Request, res: Response) {
    try {
      const { email, password }: LoginRequest = req.body;

      // Validate input
      if (!email || !password) {
        return res.status(400).json({
          success: false,
          error: 'Email and password are required'
        });
      }

      // Authenticate with Supabase
      const { data: authData, error: authError } = await supabase.auth.signInWithPassword({
        email,
        password
      });

      if (authError || !authData.user) {
        return res.status(401).json({
          success: false,
          error: 'Invalid credentials'
        });
      }

      // Get vendor profile
      const { data: vendor, error: vendorError } = await vendorService.getVendorByUserId(authData.user.id);

      if (vendorError || !vendor) {
        return res.status(500).json({
          success: false,
          error: 'Failed to retrieve vendor profile'
        });
      }

      const response: AuthResponse = {
        user: authData.user,
        vendor,
        access_token: authData.session?.access_token || '',
        refresh_token: authData.session?.refresh_token || ''
      };

      res.status(200).json({
        success: true,
        data: response,
        message: 'Login successful'
      });
      return;

    } catch (error) {
      console.error('Login error:', error);
      res.status(500).json({
        success: false,
        error: 'Login failed'
      });
      return;
    }
  }

  /**
   * Get current user profile
   * GET /api/auth/profile
   */
  async getProfile(req: Request, res: Response) {
    try {
      // User is attached by authenticateUser middleware
      const user = (req as any).user;
      
      if (!user) {
        return res.status(401).json({
          success: false,
          error: 'Authentication required'
        });
      }

      // Get vendor profile
      const { data: vendor, error: vendorError } = await vendorService.getVendorByUserId(user.id);

      if (vendorError) {
        return res.status(500).json({
          success: false,
          error: 'Failed to retrieve profile'
        });
      }

      res.status(200).json({
        success: true,
        data: {
          user,
          vendor
        }
      });
      return;

    } catch (error) {
      console.error('Get profile error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to retrieve profile'
      });
      return;
    }
  }

  /**
   * Logout user
   * POST /api/auth/logout
   */
  async logout(req: Request, res: Response) {
    try {
      const { error } = await supabase.auth.signOut();

      if (error) {
        console.error('Logout error:', error);
        return res.status(500).json({
          success: false,
          error: 'Logout failed'
        });
      }

      res.status(200).json({
        success: true,
        message: 'Logged out successfully'
      });
      return;

    } catch (error) {
      console.error('Logout error:', error);
      res.status(500).json({
        success: false,
        error: 'Logout failed'
      });
      return;
    }
  }
}

export const authController = new AuthController(); 